/**
* \file: msd_metadata.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <libudev.h>
#include <fcntl.h>
#include <linux/fs.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include "utils/logger.h"
#include "utils/helper.h"
#include "control/blacklist.h"
#include "control/configuration.h"
#include "backends/msd_events.h"
#include "backends/msd-metadata.h"
#include "backends/msd.h"
#include "backends/device_handler.h"

static error_code_t msd_metadata_extract_device_identifier(char **identifier_ptr,
		msd_uevent_info_cache_t *uevent_info_cache);

static error_code_t msd_metadata_extract_device_type(char **type_ptr,
		msd_uevent_info_cache_t *uevent_info_cache);

static error_code_t msd_metadata_extract_partition_mountsrc(char **mount_src_ptr,
		msd_uevent_info_cache_t *uevent_info_cache);

static error_code_t msd_metadata_extract_partition_mountfs(char **mount_fs_ptr,
		msd_uevent_info_cache_t *uevent_info_cache, partition_unsupported_reason_t *unsupported_reason_ptr);

static partition_unsupported_reason_t msd_metadata_determine_unsupported_reason(
		msd_uevent_info_cache_t *uevent_info_cache,	const char *fs_type);

static bool msd_metadata_check_allow_cdda_mount(msd_uevent_info_cache_t *uevent_info_cache);

static error_code_t msd_metadata_extract_partition_identifier(char **identifier_ptr,
		msd_uevent_info_cache_t *uevent_info_cache);

static error_code_t msd_metadata_extract_partition_identifier_cd(char **identifier_ptr,
		msd_uevent_info_cache_t *uevent_info_cache);

static error_code_t msd_metadata_extract_partition_identifier_disk(char **identifier_ptr,
		msd_uevent_info_cache_t *uevent_info_cache);

/*static int msd_metadata_extract_partition_number(struct udev_device *upartition);*/

static void msd_metadata_calculate_xorchecksum_over_data(char *finger_print,size_t fp_max_len,
		msd_uevent_info_cache_t *uevent_info_cache,	unsigned int dataoffset_bl, unsigned int datalen_bl);


//----------------------------MSD_Handler check for device type -----------------------------------------
bool msd_device_check_handler_type(device_t *device)
{
	return strcmp(device_get_handler_id(device),GENERAL_MSD_DEVICE)==0;
}

bool msd_partition_check_handler_type(partition_t *partition)
{
	device_t *device;
	device=partition_get_device(partition);
	return msd_device_check_handler_type(device);
}
//-------------------------------------------------------------------------------------------------------

//----------------------------MSD_Handler partition metadata members -------------------------------------
error_code_t msd_metadata_create_for_partition(partition_metadata_t **metadata_ptr,
		msd_uevent_info_cache_t *uevent_info_cache, partition_unsupported_reason_t *unsupported_reason_ptr)
{
	partition_metadata_t *metadata=NULL;
	error_code_t result;
	char *identifier="unkown";
	char *mount_src=NULL;
	char *mount_fs=NULL;
	int partition_no=1;


	result=msd_metadata_extract_partition_mountsrc(&mount_src, uevent_info_cache);

	if (result==RESULT_OK)
		result=msd_metadata_extract_partition_mountfs(&mount_fs, uevent_info_cache, unsupported_reason_ptr);

	if (result==RESULT_OK)
	{
		partition_no=msd_uevent_get_partition_number(uevent_info_cache);
		//we are accessing a device sometimes to determine the id, we are only doing so if the device is actually
		//supported
		if (*unsupported_reason_ptr==_SUPPORTED)
			result=msd_metadata_extract_partition_identifier(&identifier,uevent_info_cache);
	}

	if (result==RESULT_OK)
		metadata=partition_metadata_new(identifier, mount_src, mount_fs,partition_no);

	if (metadata==NULL)
		result=RESULT_NORESOURCE;
	else
	{
		logger_log_debug("MSD_METADATA - Following metadata attributes have been read out for partition %s:",
				msd_uevent_get_dev_node(uevent_info_cache));
		logger_log_debug("-> mount source: %s",mount_src);
		logger_log_debug("-> filesystem: %s",mount_fs);
		logger_log_debug("-> identifier: %s",identifier);
	}

	*metadata_ptr=metadata;

	return result;
}

error_code_t msd_metadata_create_for_blacklisted_partition(partition_metadata_t **metadata_ptr,
		msd_uevent_info_cache_t *uevent_info_cache, partition_unsupported_reason_t *unsupported_reason_ptr)
{
	partition_metadata_t *metadata;
	error_code_t result=RESULT_OK;
	const char *identifier="unknown";
	const char *mount_src="unknown";
	const char *mount_fs="unknown";
	int partition_no;

	partition_no=msd_uevent_get_partition_number(uevent_info_cache);
	metadata=partition_metadata_new(identifier, mount_src, mount_fs,partition_no);
	if (metadata==NULL)
		result=RESULT_NORESOURCE;

	*unsupported_reason_ptr=BLACKLISTED;
	*metadata_ptr=metadata;
	return result;
}

static error_code_t msd_metadata_extract_partition_mountsrc(char **mount_src_ptr,
		msd_uevent_info_cache_t *uevent_info_cache)

{
	*mount_src_ptr=strdup(msd_uevent_get_dev_node(uevent_info_cache));
	if (*mount_src_ptr==NULL)
		return RESULT_NORESOURCE;

	return RESULT_OK;
}

static error_code_t msd_metadata_extract_partition_mountfs(char **mount_fs_ptr,
		msd_uevent_info_cache_t *uevent_info_cache, partition_unsupported_reason_t *unsupported_reason_ptr)
{
	const char *fstype;
	fstype=msd_uevent_get_filesystem(uevent_info_cache);
	*unsupported_reason_ptr=msd_metadata_determine_unsupported_reason(uevent_info_cache, fstype);

	if (fstype==NULL)
	{
		logger_log_debug("MSD_METADATA - No file system found on %s.",
				msd_uevent_get_dev_node(uevent_info_cache));
		fstype="no file system";
	}

	*mount_fs_ptr=strdup(fstype);
	if (*mount_fs_ptr==NULL)
		return RESULT_NORESOURCE;


	return RESULT_OK;
}

static partition_unsupported_reason_t msd_metadata_determine_unsupported_reason(msd_uevent_info_cache_t *uevent_info_cache,
		const char *fs_type)
{
	partition_unsupported_reason_t unsupported_reason=_SUPPORTED;

	if (fs_type==NULL)
	{
		// we have no file system here. Do we have an audio cd here?
		if (msd_uevent_contains_cdda(uevent_info_cache))
			unsupported_reason=AUDIO_CD;
		else
			unsupported_reason=NO_FILESYSTEM;
	}
	else
	{
		//we found a file system. Check if it is supported and if we are a mixed mode cd
		//and are allowed mounting it
		if (!blacklist_is_fs_supported(fs_type))
			unsupported_reason=UNSUPPORTED_FILESYSTEM;
		if (unsupported_reason==_SUPPORTED && !msd_metadata_check_allow_cdda_mount(uevent_info_cache))
			unsupported_reason=AUDIO_MIXED_MODE;
	}

	return unsupported_reason;
}

static bool msd_metadata_check_allow_cdda_mount(msd_uevent_info_cache_t *uevent_info_cache)
{
	if (msd_uevent_contains_cdda(uevent_info_cache) && !configuration_mount_data_cdda_cds())
		return false;
	else
		return true;
}

static error_code_t msd_metadata_extract_partition_identifier(char **identifier_ptr,
		msd_uevent_info_cache_t *uevent_info_cache)
{
	error_code_t result;

	//do we have a CD here?
	if (msd_uevent_get_value_from_udevice_property_or_default(uevent_info_cache, "ID_CDROM",NULL)!=NULL)
		result=msd_metadata_extract_partition_identifier_cd(identifier_ptr,uevent_info_cache);
	else
		result=msd_metadata_extract_partition_identifier_disk(identifier_ptr,uevent_info_cache);

	return result;
}

static error_code_t msd_metadata_extract_partition_identifier_cd(char **identifier_ptr,
		msd_uevent_info_cache_t *uevent_info_cache)
{
	//format: "<DATA_TR_CNT>-<SESSION_CNT>-<LAST-SESSION-OFFSET>-<DATA HASH>"
	const char* data_tr_cnt;
	const char* session_cnt;
	const char* last_session_offset;

	//data on a cd used to calculate the md5sum (block 16 (start of isofs), 20 block length)
	const unsigned int DATA_OFFSET_BL=16;
	const unsigned int DATA_LEN_BL=10;

	//16 bytes fingerprint in hex + '\0' -> 33
	char data_finger_print[33];

	char id_buffer[2048];

	data_tr_cnt=msd_uevent_get_value_from_udevice_property_or_default(uevent_info_cache, "ID_CDROM_MEDIA_TRACK_COUNT_DATA","0");
	session_cnt=msd_uevent_get_value_from_udevice_property_or_default(uevent_info_cache,"ID_CDROM_MEDIA_SESSION_COUNT","0");
	last_session_offset=msd_uevent_get_value_from_udevice_property_or_default(uevent_info_cache,"ID_CDROM_MEDIA_SESSION_LAST_OFFSET",
				"00000000000000000000");

	msd_metadata_calculate_xorchecksum_over_data(data_finger_print, sizeof(data_finger_print),uevent_info_cache, DATA_OFFSET_BL, DATA_LEN_BL);

	snprintf(id_buffer,2048,"%s-%s-%s-%s",data_tr_cnt,session_cnt,last_session_offset,data_finger_print);

	*identifier_ptr=strdup(id_buffer);
	if (*identifier_ptr==NULL)
		return RESULT_NORESOURCE;

	return RESULT_OK;
}

static error_code_t msd_metadata_extract_partition_identifier_disk(char **identifier_ptr,
		msd_uevent_info_cache_t *uevent_info_cache)
{
	const char *uuid;

	//uuid=msd_uevent_get_value_from_udevice_property_or_default(uevent_info_cache,"ID_FS_UUID",NULL);
	uuid=msd_uevent_get_filesystem_uuid(uevent_info_cache);

	//if not, we have no idea anymore
	if (uuid==NULL)
		uuid="unknown";

	*identifier_ptr=strdup(uuid);
	if (*identifier_ptr==NULL)
		return RESULT_NORESOURCE;

	return RESULT_OK;
}

//--------------------------------------------------------------------------------------------------------

//----------------------------MSD_Handler device metadata members ----------------------------------------
error_code_t msd_metadata_create_for_device(device_metadata_t **metadata_ptr, msd_uevent_info_cache_t *uevent_info_cache)
{
	device_metadata_t *metadata=NULL;
	error_code_t result;
	char *device_identifier=NULL;
	char *device_type=NULL;

	result=msd_metadata_extract_device_identifier(&device_identifier, uevent_info_cache);

	if (result==RESULT_OK)
		result=msd_metadata_extract_device_type(&device_type, uevent_info_cache);

	if (result==RESULT_OK)
		metadata=device_metadata_new(device_type, device_identifier);

	if (metadata==NULL)
		result=RESULT_NORESOURCE;

	*metadata_ptr=metadata;

	return result;
}

static error_code_t msd_metadata_extract_device_identifier(char **identifier_ptr,
		msd_uevent_info_cache_t *uevent_info_cache)
{
	//format: "<ID_SERIAL>"
	const char* id_serial;
	id_serial=msd_uevent_get_value_from_udevice_property_or_default(uevent_info_cache,"ID_SERIAL","unkown");
	*identifier_ptr=strdup(id_serial);

	if (*identifier_ptr==NULL)
		return RESULT_NORESOURCE;

	return RESULT_OK;
}

static error_code_t msd_metadata_extract_device_type(char **type_ptr,
		msd_uevent_info_cache_t *uevent_info_cache)
{
	const char *type;
	type=msd_uevent_get_value_from_udevice_property_or_default(uevent_info_cache, "ID_TYPE",NULL);

	if (type==NULL)
	{
		//ID_TYPE is only set for devices connected via usb, apply a heuristic for others
		if (msd_uevent_get_value_from_udevice_property_or_default(uevent_info_cache, "ID_CDROM",NULL)!=NULL)
			type="cd";
		else
			type="disk";
	}

	*type_ptr=strdup(type);
	if (*type_ptr==NULL)
		return RESULT_NORESOURCE;

	return RESULT_OK;
}
//--------------------------------------------------------------------------------------------------------

//------------------------------------ read out device specifics -----------------------------------------

void msd_metadata_create_mountpoint_from_partition(msd_uevent_info_cache_t *uevent_info_cache,
		char *mount_point, size_t buf_len)
{
	strncpy(mount_point,msd_uevent_get_dev_node(uevent_info_cache),buf_len);
	helper_strcrplc(mount_point,'/','_',buf_len);
}

struct udev_device *msd_metadata_determine_parent_device(struct udev_device *upartition)
{
	const char *dev_type;

	struct udev_device *udevice;
	udevice=udev_device_get_parent(upartition);

	if (udevice!=NULL)
	{
		dev_type=udev_device_get_devtype(udevice);
		if (dev_type!=NULL)
		{
			if (strcmp(dev_type,"disk")!=0)
				udevice=NULL;
		}
		else
			udevice=NULL;
	}
	return udevice;
}

struct udev_device *msd_metadata_find_udevice_by_devnode(struct udev *udev, const char *dev_node)
{
	//taken from systemd's udevadm (udevadm-info.c)
	struct stat statbuf;
	char type;

	if (stat(dev_node, &statbuf) < 0)
		return NULL;

	if (S_ISBLK(statbuf.st_mode))
		type = 'b';
	else if (S_ISCHR(statbuf.st_mode))
		type = 'c';
	else
		return NULL;

	return udev_device_new_from_devnum(udev, type, statbuf.st_rdev);
}

int msd_metadata_get_intvalue_from_udevice_property(struct udev_device *udevice, const char *property_name, int default_val)
{
	int intval=default_val;

	const char *value_str;
	char *value_str_ret;

	value_str=udev_device_get_property_value(udevice, property_name);
	if (value_str!=NULL)
	{
		intval=strtol(value_str,&value_str_ret,10);
		//unable to parse the number
		if (value_str==value_str_ret)
			intval=default_val;
	}

	return intval;
}

const char *msd_metadata_get_value_from_udevice_property_or_default(struct udev_device *udevice, const char *property_name, const char *default_val)
{
	const char *result;
	result=udev_device_get_property_value(udevice,property_name);
	if (result==NULL)
		return default_val;
	return result;
}

static void msd_metadata_calculate_xorchecksum_over_data(char *finger_print, size_t fp_max_len,msd_uevent_info_cache_t *uevent_info_cache,
		unsigned int dataoffset_bl, unsigned int datalen_bl)
{
	error_code_t result=RESULT_OK;
	size_t blockSize=0;
	int fd;
	const char *dev_node;
	void *data=NULL;
	size_t data_len_should;
	ssize_t data_len_is=-1;
	unsigned int data_off;

	dev_node=msd_uevent_get_dev_node(uevent_info_cache);
	fd = open(dev_node,O_RDONLY);
	if (fd<0)
	{
		logger_log_error("MSD_METADATA - Error opening the device node: %s",
						msd_uevent_get_dev_node(uevent_info_cache));
		result=RESULT_READ_DEV_ERR;
	}

	if (result==RESULT_OK)
	{
		if (ioctl(fd, BLKSSZGET, &blockSize)<0)
		{
			logger_log_error("MSD_METADATA - Error getting bock size of device node: %s",
								msd_uevent_get_dev_node(uevent_info_cache));
			result=RESULT_READ_DEV_ERR;
		}
	}

	if (blockSize==0)
		blockSize=2024;

	data_off=blockSize*dataoffset_bl;
	data_len_should=blockSize*datalen_bl;

	if (result==RESULT_OK)
	{
		if (lseek(fd,(__off_t)data_off,SEEK_SET)<0)
		{
			logger_log_error("MSD_METADATA - Error seeking to data block of device node: %s",
								msd_uevent_get_dev_node(uevent_info_cache));
			result=RESULT_READ_DEV_ERR;;
		}
	}

	if (result==RESULT_OK)
	{
		data=malloc(data_len_should);
		if (data==NULL)
		{
			logger_log_error("MSD_METADATA - Error allocating a buffer of %d bytes to read out device: %s",data_len_should,
								msd_uevent_get_dev_node(uevent_info_cache));
			result=RESULT_READ_DEV_ERR;
		}
	}

	if (result==RESULT_OK)
	{
		data_len_is=read(fd,data,data_len_should);
		if (data_len_is<(ssize_t)data_len_should)
		{
			logger_log_error("MSD_METADATA - Error reading out %d bytes from device: %s, got only %d bytes.",data_len_should,
								msd_uevent_get_dev_node(uevent_info_cache),data_len_is);
			result=RESULT_READ_DEV_ERR;
		}
	}

	if (fd >= 0)
		close(fd);

	if (result==RESULT_OK && data != NULL)
	{
		int *data_int_ptr;
		int finger_print_bin[4]={0,0,0,0};
		ssize_t i=sizeof(finger_print_bin);
		int j;

		data_int_ptr=(int *)data;

		while (i<=data_len_is)
		{
			for (j=0;j<4;j++)
				finger_print_bin[j] ^= data_int_ptr[j];
			data_int_ptr+=4;
			i+=sizeof(finger_print_bin);
		}
		snprintf(finger_print, fp_max_len,"%08X%08X%08X%08X",finger_print_bin[0],finger_print_bin[1],finger_print_bin[2],finger_print_bin[3]);
	}

	if (result!=RESULT_OK)
		strncpy(finger_print,"DEV_READ_FOR_MD5_ERROR",fp_max_len);

	if (data!=NULL)
		free(data);
}

bool msd_metadata_check_contains_audio_cdda(struct udev_device *upartition)
{
	return udev_device_get_property_value(upartition, "ID_CDROM_MEDIA_TRACK_COUNT_AUDIO")!=NULL;
}
//--------------------------------------------------------------------------------------------------------
